/**
 * Copyright 2009 Roland Foerther, Carl-Eric-Menzel, Olaf Siefart
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package beispiel.clickfinder;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;

import junit.framework.TestCase;

import org.apache.wicket.ajax.AjaxRequestTarget;
import org.apache.wicket.markup.html.WebPage;
import org.apache.wicket.model.IModel;
import org.apache.wicket.model.Model;
import org.apache.wicket.util.convert.IConverter;
import org.apache.wicket.util.tester.WicketTester;

public class ClickFinderOutsideFormTest extends TestCase {
	public static class TestPage extends WebPage {
		private String lastInput;

		public IModel<Value> getModel() {
			return (IModel<Value>) getDefaultModel();
		}

		public TestPage() {
			super(new Model<Value>());
			add(new ClickFinder<Value>("clickfinder", null, Arrays.asList("A", "B", "C"), Value.class) {
				@Override
				protected List<String> getPrefixesStartingWith(final String prefix) {
					return Arrays.asList(prefix + "A", prefix + "B", prefix + "C");
				}

				@Override
				protected void onInputChanged(final AjaxRequestTarget target) {
					assertNotNull(target);
					lastInput = this.getResult();
				}

				@Override
				public IConverter getConverter(final Class<?> type) {
					assert type.equals(Value.class);
					return new IConverter() {
						public String convertToString(final Object value, final Locale locale) {
							return value.toString();
						}

						public Object convertToObject(final String value, final Locale locale) {
							return new Value(value);
						}
					};
				}
			});
		}
	}

	private WicketTester tester;
	private TestPage page;
	private ClickFinder<Value> clickFinder;

	public void testFormComponent() throws Exception {
		tester = new WicketTester();
		tester.startPage(TestPage.class);
		page = (TestPage) tester.getLastRenderedPage();
		clickFinder = (ClickFinder<Value>) tester.getComponentFromLastRenderedPage("clickfinder");
		tester.executeAjaxEvent("clickfinder:prefixes:0:prefixLink", "onclick");
		assertInput("A", false);
		tester.executeAjaxEvent("clickfinder:prefixes:1:prefixLink", "onclick");
		assertInput("AB", false);
		clickFinder.setModel(page.getModel());
		tester.executeAjaxEvent("clickfinder:prefixes:1:prefixLink", "onclick");
		assertInput("ABB", true);
		tester.executeAjaxEvent("clickfinder:previous", "onclick");
		assertInput("AB", true);
		tester.executeAjaxEvent("clickfinder:next", "onclick");
		assertInput("ABB", true);
	}

	private void assertInput(final String string, final boolean assertModel) {
		assertEquals(string, clickFinder.getResult());
		assertEquals(string, page.lastInput);
		if (assertModel) {
			assertEquals(string, clickFinder.getDefaultModelObject().toString());
			assertEquals(new Value(string), clickFinder.getDefaultModelObject());
		}
		tester.assertLabel("clickfinder:display", string);
		page.lastInput = null; // cleanup
	}
}
