/**
 * Copyright 2009 Roland Foerther, Carl-Eric-Menzel, Olaf Siefart
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package beispiel.clickfinder;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;

import junit.framework.TestCase;

import org.apache.wicket.ajax.AjaxRequestTarget;
import org.apache.wicket.markup.html.WebPage;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.form.Form;
import org.apache.wicket.model.CompoundPropertyModel;
import org.apache.wicket.model.Model;
import org.apache.wicket.model.PropertyModel;
import org.apache.wicket.util.convert.IConverter;
import org.apache.wicket.util.tester.FormTester;
import org.apache.wicket.util.tester.WicketTester;

public class ClickFinderInsideFormTest extends TestCase {
	private TestPage page;
	private WicketTester tester;
	private ClickFinder<Value> clickFinder;

	public static class TestPage extends WebPage {
		private Value value;
		private String lastInput;

		public Value getValue() {
			return value;
		}

		public void setValue(final Value value) {
			this.value = value;
		}

		public TestPage() {
			super(new Model<Value>());
			final Form<TestPage> form = new Form<TestPage>("form", new CompoundPropertyModel<TestPage>(this));
			add(form);
			form.add(new ClickFinder<Value>("value", Arrays.asList("A", "B", "C")) {
				@Override
				protected List<String> getPrefixesStartingWith(final String prefix) {
					return Arrays.asList(prefix + "A", prefix + "B", prefix + "C");
				}

				@Override
				protected void onInputChanged(final AjaxRequestTarget target) {
					assertNotNull(target);
					lastInput = this.getResult();
				}

				@Override
				public IConverter getConverter(final Class<?> type) {
					assert type.equals(Value.class);
					return new IConverter() {
						public String convertToString(final Object value, final Locale locale) {
							return value.toString();
						}

						public Object convertToObject(final String value, final Locale locale) {
							return new Value(value);
						}
					};
				}
			});
			add(new Label("output", new PropertyModel<Value>(this, "value")));
		}
	}

	public void testFormComponent() throws Exception {
		tester = new WicketTester();
		tester.startPage(TestPage.class);
		clickFinder = (ClickFinder<Value>) tester.getComponentFromLastRenderedPage("form:value");
		page = (TestPage) tester.getLastRenderedPage();
		tester.executeAjaxEvent("form:value:prefixes:0:prefixLink", "onclick");
		assertInput("A");
		assertNull(clickFinder.getDefaultModelObject());
		tester.executeAjaxEvent("form:value:prefixes:1:prefixLink", "onclick");
		assertNull(clickFinder.getDefaultModelObject());
		assertInput("AB");
		tester.executeAjaxEvent("form:value:prefixes:1:prefixLink", "onclick");
		assertNull(clickFinder.getDefaultModelObject());
		assertInput("ABB");
		tester.executeAjaxEvent("form:value:previous", "onclick");
		assertNull(clickFinder.getDefaultModelObject());
		assertInput("AB");
		tester.assertLabel("form:value:display", "AB");
		tester.executeAjaxEvent("form:value:next", "onclick");
		assertNull(clickFinder.getDefaultModelObject());
		assertInput("ABB");
		final FormTester ft = tester.newFormTester("form", false);
		ft.submit();
		assertEquals("ABB", clickFinder.getDefaultModelObject().toString());
		assertEquals(new Value("ABB"), clickFinder.getDefaultModelObject());
		tester.assertLabel("output", "ABB");
	}

	private void assertInput(final String string) {
		assertEquals(string, clickFinder.getResult());
		assertEquals(string, page.lastInput);
		tester.assertLabel("form:value:display", string);
		page.lastInput = null; // cleanup
	}
}
