/**
 * Copyright 2009 Roland Foerther, Carl-Eric-Menzel, Olaf Siefart
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package beispiel.clickfinder;

import java.util.ArrayList;
import java.util.List;

import org.apache.wicket.IRequestTarget;
import org.apache.wicket.ajax.AjaxRequestTarget;
import org.apache.wicket.ajax.markup.html.AjaxLink;
import org.apache.wicket.markup.html.CSSPackageResource;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.form.Form;
import org.apache.wicket.markup.html.form.FormComponentPanel;
import org.apache.wicket.markup.html.list.ListItem;
import org.apache.wicket.markup.html.list.ListView;
import org.apache.wicket.model.IModel;
import org.apache.wicket.model.IObjectClassAwareModel;
import org.apache.wicket.model.PropertyModel;
import org.apache.wicket.model.util.ListModel;

public abstract class ClickFinder<T> extends FormComponentPanel<T> {
	private static final long serialVersionUID = 1L;
	private final List<String> previousValues = new ArrayList<String>();
	private final List<String> nextValues = new ArrayList<String>();
	private IModel<List<String>> prefixesModel;
	private String result;

	public String getResult() {
		return result;
	}

	public ClickFinder(final String id, final List<String> initialPrefixes) {
		super(id);
		init(initialPrefixes);
	}

	public ClickFinder(final String id, final IModel<T> model, final List<String> initialPrefixes, final Class<T> conversionType) {
		super(id, model);
		setType(conversionType);
		init(initialPrefixes);
	}

	@SuppressWarnings("serial")
	private void init(final List<String> initialPrefixes) {
		this.setOutputMarkupId(true);
		this.setOutputMarkupPlaceholderTag(true);
		add(new Label("display", new PropertyModel<String>(this, "result")));
		add(new AjaxLink("previous") {
			@Override
			public void onClick(final AjaxRequestTarget target) {
				ClickFinder.this.previous();
				target.addComponent(ClickFinder.this);
			}

			@Override
			public boolean isVisible() {
				return !previousValues.isEmpty();
			}
		});
		add(new AjaxLink("next") {
			@Override
			public void onClick(final AjaxRequestTarget target) {
				ClickFinder.this.next();
				target.addComponent(ClickFinder.this);
			}

			@Override
			public boolean isVisible() {
				return !nextValues.isEmpty();
			}
		});
		this.prefixesModel = new ListModel<String>(initialPrefixes);
		final ListView<String> prefixes = new ListView<String>("prefixes", prefixesModel) {
			@Override
			protected void populateItem(final ListItem<String> item) {
				final AjaxLink<String> prefixLink = new AjaxLink<String>("prefixLink", item.getModel()) {
					@Override
					public void onClick(final AjaxRequestTarget target) {
						ClickFinder.this.input(this.getModelObject());
						target.addComponent(ClickFinder.this);
					}
				};
				item.add(prefixLink);
				prefixLink.add(new Label("prefix", item.getModel()));
			}
		};
		add(prefixes);
	}

	protected void previous() {
		if (!previousValues.isEmpty()) {
			nextValues.add(getResult());
			setResult(pop(previousValues));
			updateModelIfNotInForm();
			notifyListener();
		}
	}

	protected void next() {
		if (!nextValues.isEmpty()) {
			previousValues.add(getResult());
			setResult(pop(nextValues));
			updateModelIfNotInForm();
			notifyListener();
		}
	}

	private void notifyListener() {
		final IRequestTarget target = getRequestCycle().getRequestTarget();
		if (target instanceof AjaxRequestTarget) {
			onInputChanged((AjaxRequestTarget) target);
		} else {
			onInputChanged(null);
		}
	}

	private String pop(final List<String> stack) {
		return stack.remove(stack.size() - 1);
	}

	protected void input(final String value) {
		previousValues.add(getResult());
		setResult(value);
		nextValues.clear();
		updateModelIfNotInForm();
		notifyListener();
	}

	private void updateModelIfNotInForm() {
		if ((Form.findForm(this) == null) && (getModel() != null)) {
			convertInput();
			updateModel();
		}
	}

	protected void onInputChanged(final AjaxRequestTarget target) {
		// Kann von benutzenden Anwendungen ueberschrieben
		// werden
	}

	protected void setResult(String value) {
		if (value == null) {
			value = "";
		}
		this.result = value;
		prefixesModel.setObject(getPrefixesStartingWith(value));
	}

	@Override
	protected void convertInput() {
		final T converted = (T) getConverter(getConversionType()).convertToObject(getResult(), getLocale());
		setConvertedInput(converted);
	}

	private Class<T> getConversionType() {
		Class<T> type = getType();
		if (type == null) {
			if (getModel() instanceof IObjectClassAwareModel) {
				type = ((IObjectClassAwareModel) getModel()).getObjectClass();
			} else if (getModelObject() != null) {
				type = (Class<T>) getModelObject().getClass();
			}
			setType(type);
		}
		return type;
	}

	public ClickFinder<T> addDefaultCss() {
		this.add(CSSPackageResource.getHeaderContribution(ClickFinder.class, "ClickFinder.css"));
		return this;
	}

	protected abstract List<String> getPrefixesStartingWith(String prefix);
}
