/**
 * Copyright 2009 Roland Foerther, Carl-Eric-Menzel, Olaf Siefart
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package beispiel.clickfinder;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.SortedSet;
import java.util.TreeSet;

import org.apache.wicket.PageParameters;
import org.apache.wicket.markup.html.WebPage;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.form.Form;
import org.apache.wicket.model.CompoundPropertyModel;
import org.apache.wicket.model.IModel;
import org.apache.wicket.model.Model;
import org.apache.wicket.model.PropertyModel;
import org.apache.wicket.util.convert.IConverter;

/**
 * Homepage
 */
public class HomePage extends WebPage {
	private static final SortedSet<String> STREETS = new TreeSet<String>();
	static {
		final BufferedReader reader = new BufferedReader(new InputStreamReader(HomePage.class.getResourceAsStream("/streets.txt")));
		String line;
		try {
			while ((line = reader.readLine()) != null) {
				if (line.trim().length() > 0) {
					STREETS.add(line);
				}
			}
		} catch (final IOException e) {
			throw new RuntimeException(e);
		}
	}
	private static final long serialVersionUID = 1L;
	private final List<String> initialPrefixes = selectUniqueStreets("", 0);

	/**
	 * Constructor that is invoked when page is invoked without a session.
	 * 
	 * @param parameters
	 *            Page parameters
	 */
	public HomePage(final PageParameters parameters) {
		final IModel<String> model = new Model<String>();
		add(new ClickFinder<String>("clickfinder", initialPrefixes) {
			@Override
			protected List<String> getPrefixesStartingWith(final String prefix) {
				return selectUniqueStreets(prefix, 0);
			}
		});
		final HomePageData hpd = new HomePageData();
		final Form form = new Form<HomePageData>("form", new CompoundPropertyModel<HomePageData>(hpd));
		add(form);
		form.add(new ClickFinder<Value>("value", initialPrefixes) {
			@Override
			protected List<String> getPrefixesStartingWith(final String prefix) {
				return selectUniqueStreets(prefix, 0);
			}

			@Override
			public IConverter getConverter(final Class<?> type) {
				assert type.equals(Value.class);
				return new IConverter() {
					public String convertToString(final Object value, final Locale locale) {
						return value.toString();
					}

					public Object convertToObject(final String value, final Locale locale) {
						return new Value(value);
					}
				};
			}
		}.addDefaultCss());
		add(new Label("formValue", new PropertyModel<String>(hpd, "value")));
	}

	private List<String> selectUniqueStreets(final String prefix, final int additionalLength) {
		final SortedSet<String> set = new TreeSet<String>();
		int fullNames = 0;
		final int prefixLength = prefix.length() + 1 + additionalLength;
		boolean moreToBeHad = false;
		for (final String street : STREETS) {
			if (street.startsWith(prefix)) {
				if (prefixLength >= street.length()) {
					set.add(street);
					fullNames++;
				} else {
					set.add(street.substring(0, prefixLength));
					moreToBeHad = true;
				}
			}
		}
		if ((set.size() < 5) && moreToBeHad) {
			return selectUniqueStreets(prefix, additionalLength + 1);
		} else {
			return new ArrayList<String>(set);
		}
	}

	private List<String> makePrefixes(final String prefix) {
		return Arrays.asList(prefix + "A", prefix + "B", prefix + "C", prefix + "D", prefix + "E", prefix + "F");
	}

	public static class HomePageData implements Serializable {
		private Value value;

		public Value getValue() {
			return value;
		}

		public void setValue(final Value value) {
			this.value = value;
		}
	}

	public static class Value implements Serializable {
		private final String v;

		public Value(final String v) {
			this.v = v;
		}

		@Override
		public String toString() {
			return v == null ? "[null]" : v;
		}
	}
}
